# Generates "mongo.h" config header file containing feature flags generated by checking for the availability of certain compiler features.
# This script is invoked by the Bazel build system to generate the "mongo.h" file automatically as part of the build.
# Example usage:
# python generate_config_header.py \
#    --compiler-path /usr/bin/gcc --compiler-args "-O2 -Wall" \
#    --output-path mongo.h --template-path mongo.h.in \
#    --check-path mongo_checks.py --log-path mongo.h.log
import argparse
import inspect
import os
import sys
import textwrap
from typing import Dict


def write_config_header(input_path: str, output_path: str, definitions: Dict[str, str]) -> None:
    with open(input_path) as in_file:
        content = in_file.read()

    with open(output_path, "w", newline="\n") as file:
        output_content = content

        for key, value in definitions.items():
            output_content = output_content.replace(key, value)
        file.write(output_content)


if __name__ == "__main__":
    parser = argparse.ArgumentParser(description="Generate a config header file")
    parser.add_argument("--compiler-path", help="Path to the compiler executable", required=True)
    parser.add_argument("--compiler-args", help="Extra compiler arguments", required=True)
    parser.add_argument("--env-vars", help="Extra environment variables", required=True)
    parser.add_argument(
        "--output-path", help="Path to the output config header file", required=True
    )
    parser.add_argument(
        "--template-path", help="Path to the config header's template file", required=True
    )
    parser.add_argument("--extra-definitions", help="Extra header definitions")
    parser.add_argument("--check-path", help="Path to the suppored configure checks", required=True)
    parser.add_argument("--log-path", help="Path to the suppored configure checks", required=True)
    parser.add_argument("--additional-input", help="extra files", action="append")

    args = parser.parse_args()

    sys.path.append(".")
    generate_config_module = os.path.splitext(args.check_path)[0].replace("/", ".")
    module = __import__(generate_config_module, fromlist=["generate_config_header"])
    generate_config_header_func = getattr(module, "generate_config_header")
    try:
        generate_config_header_args = {
            "compiler_path": args.compiler_path,
            "compiler_args": args.compiler_args,
            "env_vars": args.env_vars,
            "logpath": args.log_path,
            "additional_inputs": args.additional_input,
            "extra_definitions": args.extra_definitions,
        }
        definitions = generate_config_header_func(**generate_config_header_args)
        write_config_header(args.template_path, args.output_path, definitions)
    except TypeError as exc:
        called_args = inspect.getfullargspec(generate_config_header_func).args
        print(
            textwrap.dedent(
                """\
                              
            ERROR: called import config header generator:
                    %s
                imported "generate_config_header" requires has these args:
                    %s
                but was passed these args:
                    %s
            
            """
                % (generate_config_module, called_args, list(generate_config_header_args.keys()))
            )
        )

        raise exc
