/**
 *    Copyright (C) 2025-present MongoDB, Inc.
 *
 *    This program is free software: you can redistribute it and/or modify
 *    it under the terms of the Server Side Public License, version 1,
 *    as published by MongoDB, Inc.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    Server Side Public License for more details.
 *
 *    You should have received a copy of the Server Side Public License
 *    along with this program. If not, see
 *    <http://www.mongodb.com/licensing/server-side-public-license>.
 *
 *    As a special exception, the copyright holders give permission to link the
 *    code of portions of this program with the OpenSSL library under certain
 *    conditions as described in each individual source file and distribute
 *    linked combinations including the program with the OpenSSL library. You
 *    must comply with the Server Side Public License in all respects for
 *    all of the code used other than as permitted herein. If you modify file(s)
 *    with this exception, you may extend this exception to your version of the
 *    file(s), but you are not obligated to do so. If you do not wish to do so,
 *    delete this exception statement from your version. If you delete this
 *    exception statement from all source files in the program, then also delete
 *    it in the license file.
 */
#pragma once

#include "mongo/db/extension/sdk/extension_status.h"

namespace mongo::extension::host {
/**
 * ExtensionStatusHandle is an owned handle wrapper around a
 * MongoExtensionStatus. Typically this is a handle around a MongoExtensionStatus allocated by the
 * extension whose ownership has been transferred to the host.
 *
 * Note, for the time being, we are building and linking the C++ SDK into the host API to minimize
 * code duplication. Once we are ready to decouple the C++ SDK from the server, we will need to
 * provide an implementation of the classes we are leveraging here from the SDK within the host side
 * of the API.
 *
 * ExtensionStatusHandle will eventually diverge from the SDK's implementation.
 * In some cases, a call to the extension may try to propagate a status that was originally
 * generated by the host, if a call back into the host was made by the extension. We will rely on
 * checking the VTABLE to check if the assertion came from the Host, in which case we could safely
 * rethrow the exception.
 */
using ExtensionStatusHandle = sdk::HostStatusHandle;

/**
 * HostStatus is an abstraction for providing status of calls from the extension back into the
 * host. This serves two purposes:
 * 1)  Callbacks into host code like source->getNext() may throw an exception but the calling
 *     extension code is expecting a MongoExtensionStatus object in their interface.
 * 2)  If a call into extension code yields a status of this type (as determined by vtable) then
 *     we can re-throw the exception without unwinding through the stack of the extension, which
 *     may otherwise be undefined behavior.
 *
 * For the time being, we are building and linking the C++ SDK into the host API to minimize code
 * duplication. However, once we are ready to decouple the C++ SDK from the server, we will need to
 * provide an implementation of the classes we are leveraging here from the SDK within the host side
 * of the API.
 *
 * Note, in the near future, HostStatus will diverge from sdk::ExtensionStatus, as HostStatus will
 * allow rethrowing exceptions that are coming from the Host, that have been passed through the
 * Extension as outlined in the comment above.
 *
 */
using HostStatus = sdk::ExtensionStatus;

/**
 * HostStatusOK is an ExtensionStatus without errors.
 *
 */
using HostStatusOK = sdk::ExtensionStatusOK;
}  // namespace mongo::extension::host
