/**
 *    Copyright (C) 2023-present MongoDB, Inc.
 *
 *    This program is free software: you can redistribute it and/or modify
 *    it under the terms of the Server Side Public License, version 1,
 *    as published by MongoDB, Inc.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    Server Side Public License for more details.
 *
 *    You should have received a copy of the Server Side Public License
 *    along with this program. If not, see
 *    <http://www.mongodb.com/licensing/server-side-public-license>.
 *
 *    As a special exception, the copyright holders give permission to link the
 *    code of portions of this program with the OpenSSL library under certain
 *    conditions as described in each individual source file and distribute
 *    linked combinations including the program with the OpenSSL library. You
 *    must comply with the Server Side Public License in all respects for
 *    all of the code used other than as permitted herein. If you modify file(s)
 *    with this exception, you may extend this exception to your version of the
 *    file(s), but you are not obligated to do so. If you do not wish to do so,
 *    delete this exception statement from your version. If you delete this
 *    exception statement from all source files in the program, then also delete
 *    it in the license file.
 */

#pragma once

#include "mongo/db/hasher.h"
#include "mongo/db/matcher/expression_leaf.h"

namespace mongo {

/**
 * An internal match expression node which represents a field path's hash being equal to the given
 * NumberLong constant. This expression is not intended to be written directly by users, but rather
 * generated by a rewrite of {$expr: {$eq: [{$toHashedIndexKey: '$fieldPath'}, NumberLong(123)]}}.
 * The purpose of this is to allow us to generate index bounds on a hashed index to satisfy such a
 * query.
 */
class InternalEqHashedKey : public ComparisonMatchExpressionBase {
public:
    static constexpr StringData kName = "$_internalEqHash"_sd;

    InternalEqHashedKey(boost::optional<StringData> path, long long hashVal)
        : ComparisonMatchExpressionBase(MatchType::INTERNAL_EQ_HASHED_KEY,
                                        path,
                                        Value(hashVal),
                                        // Match the agg expression semantics for traversals.
                                        ElementPath::LeafArrayBehavior::kNoTraversal,
                                        ElementPath::NonLeafArrayBehavior::kMatchSubpath) {}

    InternalEqHashedKey(std::string path, long long hashVal)
        : InternalEqHashedKey(boost::optional<StringData>(path), hashVal) {}

    InternalEqHashedKey(boost::optional<StringData> path, BSONElement value)
        // Checking the type should happen earlier during parsing.
        : InternalEqHashedKey(path, value.numberLong()) {}

    std::unique_ptr<MatchExpression> clone() const final {
        auto clone = std::make_unique<InternalEqHashedKey>(path(), _rhs);
        clone->setCollator(_collator);
        if (getTag()) {
            clone->setTag(getTag()->clone());
        }
        return clone;
    }

    StringData name() const final {
        return kName;
    }

    void acceptVisitor(MatchExpressionMutableVisitor* visitor) final {
        visitor->visit(this);
    }

    void acceptVisitor(MatchExpressionConstVisitor* visitor) const final {
        visitor->visit(this);
    }

    void appendSerializedRightHandSide(BSONObjBuilder* bob,
                                       const SerializationOptions& opts = {},
                                       bool includePath = true) const final {
        opts.appendLiteral(bob, name(), _rhs, Value((long long)1));
    }
};

}  // namespace mongo
