/**
 *    Copyright (C) 2022-present MongoDB, Inc.
 *
 *    This program is free software: you can redistribute it and/or modify
 *    it under the terms of the Server Side Public License, version 1,
 *    as published by MongoDB, Inc.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    Server Side Public License for more details.
 *
 *    You should have received a copy of the Server Side Public License
 *    along with this program. If not, see
 *    <http://www.mongodb.com/licensing/server-side-public-license>.
 *
 *    As a special exception, the copyright holders give permission to link the
 *    code of portions of this program with the OpenSSL library under certain
 *    conditions as described in each individual source file and distribute
 *    linked combinations including the program with the OpenSSL library. You
 *    must comply with the Server Side Public License in all respects for
 *    all of the code used other than as permitted herein. If you modify file(s)
 *    with this exception, you may extend this exception to your version of the
 *    file(s), but you are not obligated to do so. If you do not wish to do so,
 *    delete this exception statement from your version. If you delete this
 *    exception statement from all source files in the program, then also delete
 *    it in the license file.
 */

#pragma once

#include "mongo/transport/service_executor.h"
#include "mongo/util/out_of_line_executor.h"

#include <functional>

namespace mongo::transport {

class MockServiceExecutor : public ServiceExecutor {
public:
    Status start() override {
        return startCb();
    }

    std::unique_ptr<TaskRunner> makeTaskRunner() override {
        class Runner : public TaskRunner {
        public:
            explicit Runner(MockServiceExecutor* p) : _p{p} {}

            void schedule(Task task) override {
                return _p->scheduleTaskCb(std::move(task));
            }

            void runOnDataAvailable(std::shared_ptr<Session> session,
                                    Task onCompletionCallback) override {
                _p->runOnDataAvailableCb(session, std::move(onCompletionCallback));
            }

        private:
            MockServiceExecutor* _p;
        };
        return std::make_unique<Runner>(this);
    }

    Status shutdown(Milliseconds timeout) override {
        return shutdownCb(std::move(timeout));
    }

    size_t getRunningThreads() const override {
        return getRunningThreadsCb();
    }

    void appendStats(BSONObjBuilder* bob) const override {
        appendStatsCb(bob);
    }

    std::function<Status()> startCb;
    std::function<void(Task)> scheduleTaskCb;
    std::function<void(const std::shared_ptr<Session>&, Task)> runOnDataAvailableCb;
    std::function<Status(Milliseconds)> shutdownCb;
    std::function<size_t()> getRunningThreadsCb;
    std::function<void(BSONObjBuilder*)> appendStatsCb;
};

}  // namespace mongo::transport
