/*
 * The MIT License (MIT)
 *
 * Copyright (c) 2015-2019 Derick Rethans
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#ifndef __TIMELIB_PRIVATE_H__
#define __TIMELIB_PRIVATE_H__

#ifdef HAVE_TIMELIB_CONFIG_H
# include "timelib_config.h"
#endif

#ifdef HAVE_SYS_TIME_H
# include <sys/time.h>
#endif

#ifdef _WIN32
# ifdef HAVE_WINSOCK2_H
#  include <winsock2.h>
# endif
#endif

#include <string.h>

#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif

#if defined(HAVE_STDINT_H)
# include <stdint.h>
#endif

#if HAVE_UNISTD_H
# include <unistd.h>
#endif

#if HAVE_IO_H
# include <io.h>
#endif

#if HAVE_DIRENT_H
# include <dirent.h>
#endif

#include <stdio.h>
#include <limits.h>

#define TIMELIB_SECOND   1
#define TIMELIB_MINUTE   2
#define TIMELIB_HOUR     3
#define TIMELIB_DAY      4
#define TIMELIB_MONTH    5
#define TIMELIB_YEAR     6
#define TIMELIB_WEEKDAY  7
#define TIMELIB_SPECIAL  8
#define TIMELIB_MICROSEC 9

#define TIMELIB_SPECIAL_WEEKDAY                   0x01
#define TIMELIB_SPECIAL_DAY_OF_WEEK_IN_MONTH      0x02
#define TIMELIB_SPECIAL_LAST_DAY_OF_WEEK_IN_MONTH 0x03

#define TIMELIB_SPECIAL_FIRST_DAY_OF_MONTH        0x01
#define TIMELIB_SPECIAL_LAST_DAY_OF_MONTH         0x02

#define TIMELIB_TIME_PART_DONT_KEEP               0x00
#define TIMELIB_TIME_PART_KEEP                    0x01

#define MINS_PER_HOUR     60
#define SECS_PER_ERA   TIMELIB_LL_CONST(12622780800)
#define SECS_PER_DAY   86400
#define SECS_PER_HOUR   3600
#define USECS_PER_HOUR TIMELIB_LL_CONST(3600000000)

#define DAYS_PER_WEEK      7
#define DAYS_PER_YEAR    365
#define DAYS_PER_LYEAR   366
#define MONTHS_PER_YEAR   12
/* 400*365 days + 97 leap days */
#define DAYS_PER_ERA  146097
#define YEARS_PER_ERA    400
#define HINNANT_EPOCH_SHIFT 719468 /* 0000-03-01 instead of 1970-01-01 */

#define TIMELIB_TZINFO_PHP       0x01
#define TIMELIB_TZINFO_ZONEINFO  0x02

#define timelib_is_leap(y) ((y) % 4 == 0 && ((y) % 100 != 0 || (y) % 400 == 0))

#define TIMELIB_DEBUG(s)  if (0) { s }

#define TIMELIB_TIME_FREE(m)    \
	if (m) {        \
		timelib_free(m);    \
		m = NULL;   \
	}

#if defined (__GNUC__)
# define TIMELIB_GNUC_CHECK_VERSION(major, minor) \
      ((__GNUC__ > (major)) ||                   \
      ((__GNUC__ == (major)) && (__GNUC_MINOR__ >= (minor))))
#else
# define TIMELIB_GNUC_CHECK_VERSION(major, minor) 0
#endif

#if TIMELIB_GNUC_CHECK_VERSION(7, 0)
# define TIMELIB_BREAK_INTENTIONALLY_MISSING __attribute__ ((fallthrough));
#else
# define TIMELIB_BREAK_INTENTIONALLY_MISSING
#endif

#if defined(__has_builtin)
# if __has_builtin(__builtin_saddll_overflow)
#  define TIMELIB_HAVE_BUILTIN_SADDLL_OVERFLOW 1
# endif
#endif

#ifndef TIMELIB_HAVE_BUILTIN_SADDLL_OVERFLOW
# define TIMELIB_HAVE_BUILTIN_SADDLL_OVERFLOW 0
#endif

struct _ttinfo
{
	int32_t      offset;
	int          isdst;
	unsigned int abbr_idx;

	unsigned int isstdcnt;
	unsigned int isgmtcnt;
};

struct _tlinfo
{
	int64_t  trans;
	int32_t  offset;
};


#ifndef LONG_MAX
#define LONG_MAX 2147483647L
#endif

#ifndef LONG_MIN
#define LONG_MIN (- LONG_MAX - 1)
#endif

#ifdef __cplusplus
extern "C" {
#endif

/* From unixtime2tm.c */
int timelib_apply_localtime(timelib_time *t, unsigned int localtime);

/* From parse_posix.c */
timelib_sll timelib_ts_at_start_of_year(timelib_sll year);
ttinfo* timelib_fetch_posix_timezone_offset(timelib_tzinfo *tz, timelib_sll ts, timelib_sll *transition_time);

/* From parse_tz.c */
void timelib_time_tz_abbr_update(timelib_time* tm, const char* tz_abbr);
ttinfo* timelib_fetch_timezone_offset(timelib_tzinfo *tz, timelib_sll ts, timelib_sll *transition_time);

/* From timelib.c */
int timelib_strcasecmp(const char *s1, const char *s2);
int timelib_strncasecmp(const char *s1, const char *s2, size_t n);

#ifdef __cplusplus
} /* extern "C" */
#endif

#endif
